<?php
declare(strict_types=1);

namespace OCA\SpeakchatApi\Service;

use Exception;

class CurlHttpClient implements HttpClientInterface
{
    private const DEFAULT_HEADERS = [
        'Content-Type: application/json',
    ];

    /**
     * Sendet eine HTTP-Anfrage (GET, POST, PUT, DELETE)
     *
     * @param string $method
     * @param string $path
     * @param array|null $body
     * @return array
     */
    public function request(string $method, string $path, ?array $body = null): array
    {
        $config = \OC::$server->getConfig();
        $baseUrl = $config->getAppValue('speakchatapi', 'base_url');
        $authToken = $config->getAppValue('speakchatapi', 'auth_token');

        $url = $baseUrl . $path;

        $ch = curl_init($url);
        $headers = self::DEFAULT_HEADERS;
        $headers[] = 'X-speakchat-auth: ' . $authToken;

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, strtoupper($method));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            throw new Exception('Curl error: ' . curl_error($ch));
        }

        curl_close($ch);

        return [
            'status' => $httpCode,
            'body' => json_decode($response, true),
        ];
    }

    /**
     * Sendet eine POST-Anfrage an die angegebene URL
     *
     * @param string $url
     * @param string $authToken
     * @param string|null $payload
     * @return array
     */
    public function sendPostRequest(string $url, string $authToken, ?string $payload): array
    {
        $ch = curl_init($url);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);

        $headers = [
            'Accept: application/vnd.yclients.v2+json',
            'Authorization: Bearer ' . $authToken,
        ];

        if ($payload !== null) {
            $headers[] = 'Content-Type: application/json';
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        }

        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $response = curl_exec($ch);
        $statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            throw new Exception('Curl error: ' . curl_error($ch));
        }

        curl_close($ch);

        return [
            'status_code' => $statusCode,
            'response' => $response,
        ];
    }
}
