<?php

declare(strict_types=1);

/**
 * @copyright Copyright (c) 2024 Speakchat Team
 *
 * @author Sergej Alekseev <sergej.alekseev@googlemail.com>
 */

namespace OCA\SpeakchatApi\Service;

use OCP\IConfig;
use OCP\IURLGenerator;
use OCP\Mail\IMailer;
use OCP\IUserManager;
use OCP\IUser;
use OCP\L10N\IFactory as IL10NFactory;

use Psr\Log\LoggerInterface;

class MailService {

	public function __construct(
		private IURLGenerator $urlGenerator,
		private IMailer $mailer,
		private IConfig $config,
		private IL10NFactory $l10nFactory,
		private IUserManager $userManager,
		private LoggerInterface $logger,
		) {
	}

	/**
	 * Sends notification email to the user about the confirm success activation integration.
	 *
	 * @param  $transaction the failed transaction objet
	 */
	public function sendUserActivationSuccessEmail(IUser $user): string {
		$to = array($user->getEMailAddress() => $user->getDisplayName());

		$language = $this->l10nFactory->getUserLanguage($user);
		$l = $this->l10nFactory->get('speakchatapi', $language);

		$template = $this->mailer->createEMailTemplate('activation_integration_success', []);

		$subject = $l->t('Speakchat video conferencing integration has been activated!');

		$template->setSubject($subject);
		$template->addHeader();

		$template->addHeading($l->t('Speakchat Video Conferencing Now Available in Your YCLIENTS Account'));

//		$msg = $l->t('Dear user %s.,', [$user->getDisplayName()]);
//		$msg = $msg . '<br><br>';
		$msg = $l->t('Integration completed successfully. Speakchat video conferencing features are now available in your YCLIENTS account. You can find detailed instructions on how to create video meetings within YCLIENTS here.');
        $msg = $msg . '<br><br>';
        $msg = $msg . $l->t('What this means for you:');
        $msg = $msg . '<br><br>';
        $msg = $msg . $l->t('The ability to schedule and host video calls directly from your YCLIENTS dashboard');
        $msg = $msg . '<br><br>';
        $msg = $msg . $l->t('No need for third-party platforms – everything works in one place');
        $msg = $msg . '<br><br>';
        $msg = $msg . $l->t('Your clients will receive secure links for joining meetings automatically');

		$template->addBodyText($msg);

		//https://cloud.speakchat.ru/index.php/apps/speakchatapp/payment/success?Shp_trId=e867a4d4-98cf-4b63-84aa-0314beb68c5b
// 		$relativeLink = '/index.php/apps/speakchatapp/payment/success?Shp_trId=' . $transation->getTrId();
// 		$link = $this->urlGenerator->getAbsoluteURL($relativeLink);
		$link = 'Speakchat.ru';

//		$template->addBodyButton(
//			$l->t('Перейти!'),
//			$link
//		);

// 		$template->addFooter();

// 		$from = [$this->appConfig->fromEmailAdress() => $this->appConfig->fromName()];
		$from = ['noreply@linkchat.io'];

		$res = $this->sendEmail($from, $to, $template);

		return $res;
	}

	/**
	 * Sends notification email to the user about the failed lic activation.
	 *
	 * @param  $transaction the failed transaction objet
	 * @throws \Exception
	 */
	public function sendUserLicActivationFailedEmail($transation): string {
		$userId = $transation->getUserId();
		$user = $this->userManager->get($userId);
		$to = array($user->getEMailAddress() => $user->getDisplayName());

		$language = $this->l10nFactory->getUserLanguage($user);
		$l = $this->l10nFactory->get('speakchatapp', $language);

		$template = $this->mailer->createEMailTemplate('user_lic_activation_falied', []);

		$subject = $l->t('Your licence can not be activated automatically!');

		$template->setSubject($subject);
		$template->addHeader();

		$template->addHeading($l->t('Licence activation'));

		$msg = $l->t('Dear user %s,', [$user->getDisplayName()]);
		$msg = $msg . '<br><br>';
		$msg = $msg . $l->t('Your licence can not be activated automatically!')
				    . ' '
				    . $l->t('Check whether there is too much data in your storage and, if necessary, delete data that is no longer needed.')
				    . ' '
					. $l->t('Check which rooms are active in your account and how many rooms are included in your license. If necessary, delete rooms in your account.');

		$template->addBodyText($msg);

		//https://cloud.speakchat.ru/index.php/apps/speakchatapp/payment/success?Shp_trId=e867a4d4-98cf-4b63-84aa-0314beb68c5b
		$relativeLink = '/index.php/apps/speakchatapp/payment/success?Shp_trId=' . $transation->getTrId();
		$link = $this->urlGenerator->getAbsoluteURL($relativeLink);

		$template->addBodyButton(
			$l->t('Try again!'),
			$link
		);

		$template->addFooter();

		$from = [$this->appConfig->fromEmailAdress() => $this->appConfig->fromName()];

		$res = $this->sendEmail($from, $to, $template);

		return $res;
	}

	/**
	 * Sends update lic notification email to the user
	 *
	 * @param string $userId the id of the user to be notified
	 * @throws \Exception
	 */
	public function sendUserUpdateLicEmail(string $userId): string {

		$user = $this->userManager->get($userId);
		if(!$user) {
			return 'WARNING: User not found';
		}

		$to = array($user->getEMailAddress() => $user->getDisplayName());

		$language = $this->l10nFactory->getUserLanguage($user);
		$l = $this->l10nFactory->get('speakchatapp', $language);

		$link = $this->urlGenerator->linkToRouteAbsolute('speakchatapp.page.app', []);

		$template = $this->mailer->createEMailTemplate('user_lic_update', []);

		$subject = $l->t('Your account has been actualized');

		$template->setSubject($subject);
		$template->addHeader();

		$template->addHeading($l->t('Account actualization'));

		$msg = $l->t('Dear user %s,', [$user->getDisplayName()]);
		$msg = $msg . '<br><br>';
		$msg = $msg . $l->t('Your account has been actualized');
		$template->addBodyText($msg);

		$template->addBodyButton(
			$l->t('Go to your account'),
			$link
		);

		$template->addFooter();

		//$from = Util::getDefaultEmailAddress('register');
		//$from = ['info@linkchat.io' => 'Speakchat'];
		$from = [$this->appConfig->fromEmailAdress() => $this->appConfig->fromName()];

		$res = $this->sendEmail($from, $to, $template);

		return $res;
	}

	/**
	 * Sends expiration lic notification email to the user
	 *
	 * @param string $userId the id of the user to be notified
	 * @throws \Exception
	 */
	public function sendUserExpirationLicEmail(string $userId): string {

		$user = $this->userManager->get($userId);
		if(!$user) {
			return 'WARNING: User not found';
		}

		$to = array($user->getEMailAddress() => $user->getDisplayName());

		$language = $this->l10nFactory->getUserLanguage($user);
		$l = $this->l10nFactory->get('speakchatapi', $language);

		$link = $this->urlGenerator->linkToRouteAbsolute('speakchatapi.page.index', []);

		$template = $this->mailer->createEMailTemplate('user_lic_api_expiration', []);

		$subject = $l->t('Your plan has expired');

		$template->setSubject($subject);
		$template->addHeader();

		$template->addHeading($l->t('Expiration Notice'));

		$msg = $l->t('Dear user %s,', [$user->getDisplayName()]);
		$msg = $msg . '<br><br>';
		$msg = $msg . $l->t('The operation of integrated Speakchat video conferences has been suspended. Access to online meetings is restricted.');
		$msg = $msg . '<br><br>';
		$msg = $msg . $l->t('To resume using API integrations, please top up your balance.');

		$template->addBodyText($msg);

		$template->addBodyButton(
			$l->t('Go to your account'),
			$link
		);

		$template->addFooter();

		//$from = Util::getDefaultEmailAddress('register');
		//$from = ['info@linkchat.io' => 'Speakchat'];
//		$from = [$this->appConfig->fromEmailAdress() => $this->appConfig->fromName()];
        $from = ['noreply@linkchat.io'];

		$res = $this->sendEmail($from, $to, $template);

		return $res;
	}

	/**
	 * Sends reminder lic notification email to the user
	 *
	 * @param string $userId the id of the user to be notified
	 * @throws \Exception
	 */
	public function sendUserReminderLicEmail(string $userId, $expDate): string {
		$user = $this->userManager->get($userId);
		if(!$user) {
			return 'WARNING: User not found';
		}

		$to = array($user->getEMailAddress() => $user->getDisplayName());

		$language = $this->l10nFactory->getUserLanguage($user);
		$l = $this->l10nFactory->get('speakchatapi', $language);

		$link = $this->urlGenerator->linkToRouteAbsolute('speakchatapi.page.index', []);

		$template = $this->mailer->createEMailTemplate('user_lic_api_reminder', []);

		$subject = $l->t('Your plan API expires soon');

		$template->setSubject($subject);
		$template->addHeader();

		$template->addHeading($l->t('Your Speakchat API access will expire soon.'));

//		$msg = $l->t('Dear user %s,', [$user->getDisplayName()]);
//		$msg = $msg . '<br><br>';
		$msg = $l->t('This is a reminder that your account will expire at %s.', [$expDate->format('Y-m-d H:i:s')]);
		$msg = $msg . '<br><br>';
		$msg = $msg . $l->t('If you want to continue using the service, explore rates and other options, please visit our website and leave a request.');

		$template->addBodyText($msg);

		$template->addBodyButton(
			$l->t('Go to your account'),
			$link
		);

		$template->addFooter();

		//$from = Util::getDefaultEmailAddress('register');
		//$from = ['info@linkchat.io' => 'Speakchat'];
//        $from = [$this->appConfig->fromEmailAdress() => $this->appConfig->fromName()];
        $from = ['noreply@linkchat.io'];

		$res = $this->sendEmail($from, $to, $template);

		return $res;
	}

	private function sendEmail($from, $to, $template): string {
		$message = $this->mailer->createMessage();
		$message->setFrom($from);
		$message->setTo($to);
		$message->useTemplate($template);
		$failedRecipients = $this->mailer->send($message);

		$res = 'Email sent successfully!';
		if (!empty($failedRecipients)) {
			$res = 'E-mail sent failed. Recipients:  '  . print_r($failedRecipients, true);
			$this->logger->error($res);
		}

		return $res;
	}

}
