<?php

declare(strict_types=1);

namespace OCA\SpeakchatApi\Service;

class YClientsAPI
{
    private string $apiUrl = 'https://api.yclients.com/marketplace/partner/callback';
    private string $notificationUrl = 'https://api.yclients.com/marketplace/partner/payment';
    private string $refundUrl = 'https://api.yclients.com/marketplace/partner/payment/refund/';

    private HttpClientInterface $httpClient;

    public function __construct(HttpClientInterface $httpClient)
    {
        $this->httpClient = $httpClient;
    }

    /**
     * Sendet eine Callback-Anfrage an YClients.
     *
     * @param string $authToken
     * @param string $salonId
     * @param string $applicationId
     * @return array
     */
    public function sendCallback(string $authToken, string $salonId, string $applicationId): array
    {
        $payload = json_encode([
            'salon_id' => $salonId,
            'application_id' => $applicationId,
        ]);

        return $this->httpClient->sendPostRequest($this->apiUrl, $authToken, $payload);
    }

    /**
     * Sendet eine Benachrichtigung mit Zahlungsdetails an YClients.
     *
     * @param string $authToken
     * @param string $salonId
     * @param string $applicationId
     * @param float $paymentSum
     * @param string $paymentDate
     * @param string $periodFrom
     * @param string $periodTo
     * @return array
     */
    public function sendNotificationWithPayment(
        string $authToken,
        string $salonId,
        string $applicationId,
        float $paymentSum,
        string $paymentDate,
        string $periodFrom,
        string $periodTo,
    ): array {
        $data = [
            'salon_id' => $salonId,
            'application_id' => $applicationId,
            'payment_sum' => $paymentSum,
            'currency_iso' => 'RUB',
            'payment_date' => $paymentDate,
            'period_from' => $periodFrom,
            'period_to' => $periodTo,
        ];

        $payload = json_encode($data);

        return $this->httpClient->sendPostRequest($this->notificationUrl, $authToken, $payload);
    }

    /**
     * Sendet eine Rückerstattungsbenachrichtigung an YClients.
     *
     * @param string $authToken
     * @param string $paymentId
     * @return array
     */
    public function sendRefundNotification(string $authToken, string $paymentId): array
    {
        return $this->httpClient->sendPostRequest($this->refundUrl . $paymentId, $authToken, null);
    }
}
