<?php

declare(strict_types=1);

namespace OCA\SpeakchatApi\Controller;

use OCA\SpeakchatApi\Db\IntegrationMapper;
use OCA\SpeakchatApi\Db\Integration;

use OCA\SpeakchatApi\Service\CurlHttpClient;
use OCA\SpeakchatApi\Service\SpeakchatApiClient;
use OCA\SpeakchatApi\Service\YClientsAPI;
use OCA\SpeakchatApi\Service\MailService;


use OCP\AppFramework\Controller;
use OCP\AppFramework\Http;
use OCP\AppFramework\Http\DataResponse;
use OCP\IRequest;
use OCP\IUserSession;
use Ramsey\Uuid\Uuid;
use OCP\ILogger;
use OCP\IConfig;

class IntegrationApiController extends Controller {

    /**
     * Authorization token for YClients API.
     * You should store this securely, e.g., in environment variables.
     * private const SECRET_KEY_YCLIENTS = 'dsec6xk****';
     */

    /**
     * Unique application id in the market YClients.
     * You should store this securely, e.g., in environment variables.
     * private const APPLICATION_ID = '20xxx';
     */

    /**
     * $apiId -
     * Unique integration number inside speakchat
     * or unique value of ycleints client (salon_id)
     */

	 private IntegrationMapper $integrationMapper;
	 private ?string $userId;
	 private IUserSession $userSession;
	 private ILogger $logger;
	 private IConfig $config;
	 private MailService $mailService;
	 private YClientsAPI $yClientsApi;
	 private SpeakchatApiClient $speakchatApiClient;

	public function __construct(
		string $AppName,
		IRequest $request,
		IntegrationMapper $integrationMapper,
		?string $userId,
		IUserSession $userSession,
        IConfig $config,
        ILogger $logger,
		MailService $mailService,
		CurlHttpClient $curlHttpClient,
	) {
		parent::__construct($AppName, $request);
		$this->integrationMapper = $integrationMapper;
		$this->userId = $userId;
		$this->userSession = $userSession;
		$this->logger = $logger;
        $this->mailService = $mailService;
        $this->config = $config;

        $this->yClientsApi = new YClientsAPI($curlHttpClient);
        $this->speakchatApiClient = new SpeakchatApiClient($curlHttpClient);
	}

	/**
	 * @NoAdminRequired
	 */
	public function index(): DataResponse {
       	if ($this->userSession->isLoggedIn() === false) {
       		return new DataResponse(null, Http::STATUS_FORBIDDEN);
       	}
       	
       	$user = $this->userSession->getUser();
       	$integrationsApi = $this->integrationMapper->findAllByCreator($user);

       	return new DataResponse($integrationsApi);
	}

    /**
	 * @NoAdminRequired
	 */
	public function create(string $name, $apiId, $integrationType, $status): DataResponse {
		if ($this->userSession->isLoggedIn() === false) {
			return new DataResponse(null, Http::STATUS_FORBIDDEN);
		}

        $integration = $this->integrationMapper->findByApiId($apiId);
        if ($integration !==  null) {
            return new DataResponse(['status' => 'error','error' => 'Integration with this apiId already exists'], Http::STATUS_CONFLICT);
        }

		if ($integrationType === 'Yclients'){
			$keyApi = $this->config->getAppValue('speakchatapi', 'secret_key_yclients');
			$authToken = $this->config->getAppValue('speakchatapi', 'secret_key_yclients');
			$applicationId = $this->config->getAppValue('speakchatapi', 'application_id');

			// registration YClientsAPI
			$callback = $this->yClientsApi->sendCallback($authToken, $apiId, $applicationId);
			if($callback['status_code'] !== 201) {
				return new DataResponse(null, $callback['status_code']);
			}	
		}
		else {
			$keyApi = Uuid::uuid4();
		}

        // registration SpeakchatApi
        $instanceDomain = $this->config->getAppValue('speakchatapi', 'instance_domain');

		$expDate = $this->calculateExpirationDate();
		$account = [
		   'instanceId' => $apiId,
		   'apiKey' => $keyApi,
		   'instanceDomain' => $instanceDomain,
		   'expDate' => $expDate,
        ];

		$response = $this->speakchatApiClient->registerAccount($account);
		if($response['status']!== 201) {
			return new DataResponse(null, $response['status']);
		}
        $this->logger->info('Account data: ' . json_encode($account));

		$integration = new Integration();
		$integration->setName($name);
		$integration->setCreatorId($this->userId);
		$integration->setApiId($apiId);
		$integration->setKeyApi($keyApi);
		$integration->setStatus($status);
		$integration->setIntegrationType($integrationType);
		$integration->setExpDate($expDate);

		$data = $this->integrationMapper->insert($integration);

       	$user = $this->userSession->getUser();
        $this->mailService->sendUserActivationSuccessEmail($user);

		return new DataResponse($data);
	}

    /**
	 * @NoAdminRequired
	 */
	public function modify(int $id, $name, $status): DataResponse {
		if ($this->userSession->isLoggedIn() === false) {
			return new DataResponse(null, Http::STATUS_FORBIDDEN);
		}

        $integration = $this->integrationMapper->findById($id);
		if ($integration === null) {
			return new DataResponse([], Http::STATUS_NOT_FOUND);
		}

        if ($integration->getCreatorId() !== $this->userId) {
            return new DataResponse(null, Http::STATUS_FORBIDDEN);
        }

        $integration->setName($name);
        $integration->setStatus($status);

		return new DataResponse($this->integrationMapper->update($integration));
	}

    /**
	 * @NoAdminRequired
	 */
	public function delete(int $id): DataResponse {
		if ($this->userSession->isLoggedIn() === false) {
			return new DataResponse(null, Http::STATUS_FORBIDDEN);
		}

		$integration = $this->integrationMapper->findById($id);

		if ($integration === null) {
			return new DataResponse([], Http::STATUS_NOT_FOUND);
		}

        if ($integration->getCreatorId() !== $this->userId) {
            return new DataResponse(null, Http::STATUS_FORBIDDEN);
        }

        $apiId = $integration->getApiId();

		$response = $this->speakchatApiClient->deleteAccount($apiId);
		if($response['status']!== 204) {
			return new DataResponse(null, $response['status']);
		}

		$this->integrationMapper->delete($integration);
		return new DataResponse(null, Http::STATUS_NO_CONTENT);
	}

    /**
     * Innternal endpoint for the yclients integation.
     * This endpoint will be called by the uninstall webhook.
     * @param $apiId - the salon_id of the yclients integration
     * @return 204
     *
	 * @NoAdminRequired
	 * @PublicPage
	 */
	public function deleteByApiId(string $apiId): DataResponse {

		// Test auth Header
		$authHeader = $this->request->getHeader('x-speachchat-auth');
        $secretKeyYclients = $this->config->getAppValue('speakchatapi', 'secret_key_yclients');

		if ($authHeader !== $secretKeyYclients) {
			return new DataResponse(null, Http::STATUS_UNAUTHORIZED);
		}

		// Get integration using API ID
		$integration = $this->integrationMapper->findByApiId($apiId);

		if ($integration === null) {
			return new DataResponse([], Http::STATUS_NO_CONTENT);
		}

		// Delete integration
		$this->integrationMapper->delete($integration);

		$this->logger->info('deleteByApiId success: ' . $apiId);
		return new DataResponse([], Http::STATUS_NO_CONTENT);
	}

    private function calculateExpirationDate(): string
    {
        $trialDays = $this->config->getAppValue('speakchatapi', 'trial_period_days');

        $date = new \DateTime('now', new \DateTimeZone('UTC'));
        $date->modify('+' . $trialDays . ' day');
        return $date->format('Y-m-d\TH:i:s\Z');
    }
}
