<?php

declare(strict_types=1);

/**
 * Reminder service
 *
 * @copyright Copyright (c) 2024 Speakchat Team
 *
 * @author Sergej Alekseev <sergej.alekseev@googlemail.com>
 */

namespace OCA\SpeakchatApi\Service;

use OCA\SpeakchatApi\AppInfo\Application;
use OCA\SpeakchatApi\Db\IntegrationMapper;
use OCA\SpeakchatApi\Service\MailService;
use OCP\IConfig;

use Psr\Log\LoggerInterface;

class ReminderExpirationService {

	public function __construct(
		protected IntegrationMapper $integrationMapper,
		protected LoggerInterface $logger,
		protected MailService $mailService,
 		protected IConfig $appConfig,
	) {
	}

	/**
	 * @param \DateTime  $dateTime - now as a date time object
	 */
	public function executeReminders(\DateTime $dateTime): array {

		/* execute only if userLic enabled */
// 		if(!$this->appConfig->userLicEnabled()) {
// 			return array();
// 		}
//
		/* use pagination for lics iterate to save memory */
        $pageSize = 10;
        $offset = 0;
        do {
        	$allIntegrations = $this->integrationMapper->getIntegrations($offset, $pageSize);
        	foreach( $allIntegrations as $integration ) {
    			$this->checkExpiration($dateTime, $integration);
    		}

        	$offset = $offset + $pageSize;
        } while (count($allIntegrations) != 0 );

        return array();
	}

	private function checkExpiration(\DateTime $dateTime, $integration): void {

		$userId = $integration->getCreatorId();
		$reminderEmailSent = $integration->getReminderEmailSent();
		$expirationEmailSent = $integration->getExpirationEmailSent();
		$countLic = $integration->getNumberOfUsers();
		$expDate = $integration->getExpDate();

		$this->logger->info('checkExpiration for ' . $userId .
				' $reminderEmailSent: ' . $reminderEmailSent .
				' $countLic: ' . $countLic .
				' expData: ' . $expDate .
				' $expirationEmailSent: ' . $expirationEmailSent);

        if($expDate !== null) {
            $expiresAt = new \DateTime($expDate, new \DateTimeZone("UTC"));

            $this->logger->info(
                    'expiresAt: ' . $expiresAt->format('Y-m-d\TH:i:s\Z') .
                    ' $dateTime: ' . $dateTime->format('Y-m-d\TH:i:s\Z'));

            /* notify user for expiration */
            if ($expiresAt < $dateTime ) {
                if($expirationEmailSent == 0) {
                    $this->logger->info('lic expired, send email to ' . $userId );

                    $this->mailService->sendUserExpirationLicEmail($userId);

                    $integration->setExpirationEmailSent(1);
                    $integration->setStatus('disable');
                    $this->integrationMapper->update($integration);

                }
            }

            /* user reminder */
            $remindBeforeInDays = $this->appConfig->getAppValue('speakchatapi', 'reminder_period_days');
            $reminderDate = clone $dateTime;
            $reminderDate->modify('+' .$remindBeforeInDays. ' day');

            $this->logger->info(
                    '$expDate : ' . $expiresAt->format('Y-m-d\TH:i:s\Z') .
                    ' $dateTime +' .$remindBeforeInDays. ': ' . $reminderDate->format('Y-m-d\TH:i:s\Z'));

            if ($expiresAt < $reminderDate ) {
                if($reminderEmailSent == 0) {
                    $this->logger->info('lic expires soon, send email to ' . $userId );

                    $this->mailService->sendUserReminderLicEmail($userId, $expiresAt);

                    $integration->setReminderEmailSent(1);
                    $this->integrationMapper->update($integration);
                }
            }

        } else {
            $this->logger->warning('expDate is missing in the Lic object for ' . $userId);
        }

		$this->logger->info('check finished for ' . $userId );
	}
}
