<?php
declare(strict_types=1);

namespace OCA\SpeakchatApi\Service;
use \Exception;

class SpeakchatApiClient
{
	private HttpClientInterface $http;
	
    public function __construct(HttpClientInterface $http)
    {
        $this->http = $http;
    }

    /**
     * Registers a new API account in the Speakchat system.
     *
     * Sends a POST request to the /accounts endpoint with the account data.
     *
     * @param array $accountData Associative array containing:
     *                           - accountId (string)
     *                           - apiKey (string)
     *                           - instanceDomain (string)
     *                           - status (string, optional)
     *
     * @return array Response array with keys:
     *               - 'status' (int): HTTP status code
     *               - 'body' (array|null): Server response data
     */
    public function registerAccount(array $accountData): array
    {
    	 return $this->http->request('POST', '/accounts', $accountData);
    }

    /**
     * Retrieves an existing API account by its account ID.
     *
     * Sends a GET request to the /accounts/{accountId} endpoint.
     *
     * @param string $accountId The unique ID of the API account to retrieve.
     *
     * @return array Response array with keys:
     *               - 'status' (int): HTTP status code
     *               - 'body' (array|null): Account data or error information
     */
    public function getAccount(string $accountId): array
    {
    	return $this->http->request('GET', "/accounts/$accountId");
    }

    /**
     * Updates an existing API account by its account ID.
     *
     * Sends a PUT request to the /accounts/{accountId} endpoint with updated data.
     *
     * @param string $accountId The ID of the account to update.
     * @param array  $updateData Associative array with optional fields:
     *                           - apiKey (string)
     *                           - instanceDomain (string)
     *                           - status (string)
     *
     * @return array Response array with keys:
     *               - 'status' (int): HTTP status code
     *               - 'body' (array|null): Updated account data or error information
     */
    public function updateAccount(string $accountId, array $updateData): array
    {
    	return $this->http->request('PUT', "/accounts/$accountId", $updateData);
    }

    /**
     * Deletes an existing API account by its account ID.
     *
     * Sends a DELETE request to the /accounts/{accountId} endpoint.
     * The operation is idempotent – repeated deletions have the same effect.
     *
     * @param string $accountId The ID of the account to delete.
     *
     * @return array Response array with keys:
     *               - 'status' (int): HTTP status code (e.g., 204 on success)
     *               - 'body' (array|null): Typically empty
     */
    public function deleteAccount(string $accountId): array
    {
    	return $this->http->request('DELETE', "/accounts/$accountId");
    }

    /**
     * Retrieves statistics for a specific API account.
     *
     * Sends a GET request to the /accounts/{accountId}/statistics endpoint.
     *
     * @param string $accountId The ID of the account for which to load statistics.
     *
     * @return array Response array with keys:
     *               - 'status' (int): HTTP status code
     *               - 'body' (array|null): Statistics data or error information
     */
    public function getAccountStatistics(string $accountId): array
    {
    	return $this->http->request('GET', "/accounts/$accountId/statistics");
    }
}
