<?php
declare(strict_types=1);

namespace OCA\SpeakchatApi\Service;

use OCA\SpeakchatApi\Db\IntegrationMapper;
use Psr\Log\LoggerInterface;
use OCP\IConfig;

class SpeakchatApiIntegrationExternalService{

    private IntegrationMapper $integrationMapper;
    private SpeakchatApiClient $speakchatApiClient;
    private LoggerInterface $logger;
    private IConfig $config;

    public function __construct(
        IntegrationMapper $mapper,
        LoggerInterface $logger,
        IConfig $config,
    ){
        $this->integrationMapper = $mapper;
        $this->logger = $logger;
        $this->config = $config;

		$curlHttpClient = new CurlHttpClient();
        $this->speakchatApiClient = new SpeakchatApiClient($curlHttpClient);
		$this->yClientsApi = new YClientsAPI($curlHttpClient);
    }

    public function updateIntegration(string $apiId, $status, $countLicense, $prolongationDate, bool $modifyLic): array
    {
        $res = [];
        try {
            $integration = $this->integrationMapper->findByApiId($apiId);
            if ($integration === null) {
                throw new \RuntimeException("Integration with apiId '$apiId' not found");
            }

            if (!$modifyLic){
                $prolongationDate = $this->calculateExpDate($integration->getExpDate(),$prolongationDate);
            }

            $integration->setStatus($status);
            $integration->setNumberOfUsers($countLicense);
            $integration->setExpDate($prolongationDate);
            $integration->setReminderEmailSent(0);
            $integration->setExpirationEmailSent(0);

            $this->integrationMapper->update($integration);

            $instanceDomain = $this->config->getAppValue('speakchatapi', 'instance_domain');
            $updateData = [
                'apiKey'  => $integration->getKeyApi(),
                'instanceDomain'  => $instanceDomain,
                'status'  => $status,
                'expDate' => $prolongationDate,
            ];

            $response = $this->speakchatApiClient->updateAccount($apiId, $updateData);

            if (!isset($response['status']) || $response['status'] !== 200) {
                throw new \RuntimeException("API update failed: " . json_encode($response));
            }

            $res[] = "Integration updated successfully for apiId $apiId";
        } catch (\Throwable $e) {
            $this->logger->error('Error updating integration', ['exception' => $e]);
            $res[] = 'Error: ' . $e->getMessage();
            return $res;
        }
        return $res;
    }

	public function sendNotificationWithPayment(
		$salonId,
		$paymentSum,
		$paymentDate,
		$periodFrom,
		$periodTo): array {

		$authToken = $this->config->getAppValue('speakchatapi', 'secret_key_yclients');
		$applicationId = $this->config->getAppValue('speakchatapi', 'application_id');

        $this->logTransactionData($authToken, $salonId, $applicationId, $paymentSum, $paymentDate, $periodFrom, $periodTo);

		return $this->yClientsApi->sendNotificationWithPayment(
		       $authToken,
			   $salonId,
			   $applicationId,
			   $paymentSum,
			   $paymentDate,
			   $periodFrom,
			   $periodTo);
	}

	public function sendRefundNotification($idRefundYclients): array {
        $authToken = $this->config->getAppValue('speakchatapi', 'secret_key_yclients');
		return $this->yClientsApi->sendRefundNotification($authToken, $idRefundYclients);
    }

	public function getExpDate($apiId): string {
		$licApi = $this->integrationMapper->findByApiId($apiId);
		$expDateLic = $licApi->getExpDate();
		return $expDateLic;
	}

    public function calculateExpDate(string $currentExpDateStr, string $newExpdateStr): string {
        if ( !str_starts_with($newExpdateStr, 'ADD_DAYS')) {
            return $newExpdateStr;
        }

        // extend the current expiration date
        $nowDate = new \DateTime('now', new \DateTimeZone("UTC"));
        $currentExpDate = new \DateTime($currentExpDateStr);

        $expDate = $nowDate > $currentExpDate ? $nowDate : $currentExpDate;

        $periodInDays = substr($newExpdateStr, 9);
        $expDate->modify('+' .$periodInDays. ' day');

        return $expDate->format('Y-m-d\TH:i:s\Z');
    }

    private function logTransactionData(
        $authToken,
        $salonId,
        $applicationId,
        $paymentSum,
        $paymentDate,
        $periodFrom,
        $periodTo
    ): void {
        $this->logger->warning('YClients API transaction data', [
            'authToken' => $authToken,
            'salonId' => $salonId,
            'applicationId' => $applicationId,
            '$paymentSum' => $paymentSum,
            '$paymentDate' => $paymentDate,
            '$periodFrom' => $periodFrom,
            '$periodTo' => $periodTo,
        ]);
    }
}
